#include <d3d9.h>
#include <d3dx9.h>

#include <algorithm>

#include "HomeworkRendererFunctions.h"
#include "Utilities.h"

struct EdgePoint
{
    float x, z, oow, fog;
    VertexData data;
};

unsigned int RasterizeScanLine(unsigned int y, EdgePoint const& e0, EdgePoint const& e1, RasterizeParams const& params, Pixel* pixels, unsigned int pixelsSize)
{
    int const x0 = max((int)floor(e0.x + 0.5f)    , (int)params.scissorX);
    int const x1 = min((int)floor(e1.x + 0.5f) - 1, (int)(params.scissorX + params.scissorWidth - 1));

    float const oodx = 1.0f / (e1.x - e0.x);

    unsigned int resultCount = 0;

    for (int x = x0; x <= x1; ++x)
    {
        if (resultCount >= pixelsSize)
        {
            throw "Rasterize: insufficient space for the result";
        }

        float const t = (float(x) + 0.5f - e0.x) * oodx;
        float const t1 = 1 - t;

        float const w = 1.0f / (e0.oow * t1 + e1.oow * t);

        float const tw  = t  * w;
        float const t1w = t1 * w;

        Pixel& p = pixels[resultCount];

        p.x    = x;
        p.y    = y;
        p.z    = e0.z    * t1  + e1.z    * t;
        p.fog  = e0.fog  * t1w + e1.fog  * tw;
        p.data = e0.data * t1w + e1.data * tw;

        ++resultCount;
    }

    return resultCount;
}

EdgePoint InterpolateEdgePoint(EdgePoint const& e0, EdgePoint const& e1, float t, float t1)
{
    EdgePoint  result;

    result.x    = e0.x    * t1 + e1.x    * t;
    result.z    = e0.z    * t1 + e1.z    * t;
    result.oow  = e0.oow  * t1 + e1.oow  * t;
    result.fog  = e0.fog  * t1 + e1.fog  * t;
    result.data = e0.data * t1 + e1.data * t;

    return result;
}

unsigned int RasterizeTrapezoid(float f0y, float f1y, EdgePoint const& e00, EdgePoint const& e10, EdgePoint const& e01, EdgePoint const& e11, RasterizeParams const& params, Pixel* pixels, unsigned int pixelsSize)
{
    int const y0 = max((int)floor(f0y + 0.5f)    , (int)params.scissorY);
    int const y1 = min((int)floor(f1y + 0.5f) - 1, (int)(params.scissorY + params.scissorHeight - 1));

    float const oody = 1.0f / (f1y - f0y);

    unsigned int resultCount = 0;

    for (int y = y0; y <= y1; ++y)
    {
        float const t = (float(y) + 0.5f - f0y) * oody;
        float const t1 = 1 - t;

        EdgePoint const e0 = InterpolateEdgePoint(e00, e10, t, t1);
        EdgePoint const e1 = InterpolateEdgePoint(e01, e11, t, t1);

        resultCount += RasterizeScanLine(y, e0, e1, params, pixels + resultCount, pixelsSize - resultCount);
    }

    return resultCount;
}

unsigned int MyRendererFunctions::Rasterize(ScreenTriangle const& triangle, RasterizeParams const& params, Pixel* pixels, unsigned int pixelsSize) const
{
    LightedVertex const* vertex[3] = { &triangle.vtx[0], &triangle.vtx[1], &triangle.vtx[2] };

    if (vertex[0]->position.y <= vertex[1]->position.y)
    {
        if (vertex[1]->position.y <= vertex[2]->position.y)
        {
            // Already ordered.
        }
        else if (vertex[0]->position.y <= vertex[2]->position.y)
        {
            std::swap(vertex[1], vertex[2]);
        }
        else
        {
            std::swap(vertex[1], vertex[2]);
            std::swap(vertex[0], vertex[1]);
        }
    }
    else
    {
        if (vertex[0]->position.y <= vertex[2]->position.y)
        {
            std::swap(vertex[0], vertex[1]);
        }
        else if (vertex[1]->position.y <= vertex[2]->position.y)
        {
            std::swap(vertex[0], vertex[1]);
            std::swap(vertex[1], vertex[2]);
        }
        else
        {
            std::swap(vertex[0], vertex[2]);
        }
    }

    float const yv0 = vertex[0]->position.y;
    float const yv1 = vertex[1]->position.y;
    float const yv2 = vertex[2]->position.y;
    EdgePoint const ev0 = { vertex[0]->position.x, vertex[0]->position.z, vertex[0]->position.w, vertex[0]->fog * vertex[0]->position.w, vertex[0]->data * vertex[0]->position.w };
    EdgePoint const ev1 = { vertex[1]->position.x, vertex[1]->position.z, vertex[1]->position.w, vertex[1]->fog * vertex[1]->position.w, vertex[1]->data * vertex[1]->position.w };
    EdgePoint const ev2 = { vertex[2]->position.x, vertex[2]->position.z, vertex[2]->position.w, vertex[2]->fog * vertex[2]->position.w, vertex[2]->data * vertex[2]->position.w };

    const float mid = (yv1 - yv0) / (yv2 - yv0);
    const float mid1 = 1 - mid;

    EdgePoint const emid = InterpolateEdgePoint(ev0, ev2, mid, mid1);

    unsigned int resultCount = 0;

    if (emid.x < ev1.x)
    {
        resultCount += RasterizeTrapezoid(yv0, yv1, ev0, emid, ev0, ev1, params, pixels, pixelsSize);
        resultCount += RasterizeTrapezoid(yv1, yv2, emid, ev2, ev1, ev2, params, pixels + resultCount, pixelsSize - resultCount);
    }
    else
    {
        resultCount += RasterizeTrapezoid(yv0, yv1, ev0, ev1, ev0, emid, params, pixels, pixelsSize);
        resultCount += RasterizeTrapezoid(yv1, yv2, ev1, ev2, emid, ev2, params, pixels + resultCount, pixelsSize - resultCount);
    }

    return resultCount;
}
